<?php

namespace App\Traits;

use Carbon\Carbon;
use App\Models\HeadGroup;
use App\Models\IncomeHead;
use App\Models\AccountType;
use App\Models\AccountingYear;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Models\Organisation;
use App\Models\AssetBroughtForward;
use App\Models\PaymentType;
use App\Models\Bank;
use App\Models\Head;

trait CommonTrait
{
    public function getPreviousFiscalYearShort()
    {
        $today = Carbon::today();
        $fiscalStartMonth = 4; // April

        if ($today->month >= $fiscalStartMonth) {
            $prevYearStart = $today->year - 1;
        } else {
            $prevYearStart = $today->year - 2;
        }

        $prevYearEnd = $prevYearStart + 1;
        // Return short format: last two digits
        return substr($prevYearStart, -2) . '-' . substr($prevYearEnd, -2);
    }

    public function getAccountType($divition_flag = null)
    {
        $query = AccountType::select('account_type_id', 'organisation_id', 'account_type_Name', 'divition_flag')
            ->where('status', 'active');

        // Apply condition only if $divition_flag is present
        if (!empty($divition_flag)) {
            $query->where('divition_flag', $divition_flag);
        }

        $accountTypes = $query->orderBy('account_type_id', 'DESC')->get();

        return $accountTypes;
    }


    public function getIncomeHead($flag = null)
    {
        $query = IncomeHead::select('*')
            ->where(['status' => 'active']);
        // Apply condition only if $flag is present
        if (!empty($flag)) {
            $query->where('income_ac_type_flg', $flag);
        }
        $income_heads = $query->orderBy('sort_order', 'ASC')->get();
        return $income_heads;
    }

    public function getGroupHead()
    {
        $query = HeadGroup::select('name', 'id')->where(['status' => 'active']);
        $headGroup = $query->orderBy('id', 'DESC')->get();
        return $headGroup;
    }

    public static function getCurrentAccountingYear()
    {
        $accountYear = AccountingYear::select(
            'accounting_year_id',
            'accounting_year_financial',
            'accounting_start_year',
            'accounting_end_year'
        )
            ->where('status', 'active')
            ->orderBy('accounting_year_id', 'DESC')
            ->first();

        if (!$accountYear) {
            return null;
        }

        $year = [
            'accounting_year_id'        => $accountYear->accounting_year_id,
            'accounting_year_financial' => $accountYear->accounting_year_financial,
            'accounting_start_year'     => $accountYear->accounting_start_year,
            'accounting_end_year'       => $accountYear->accounting_end_year,
            'account_year_prefix'       => $accountYear->accounting_start_year . '_' . $accountYear->accounting_end_year,
        ];

        return (object) $year;
    }

    public static function getYearPrefixedTables()
    {
        $currentYear = self::getCurrentAccountingYear();
        if (!$currentYear) {
            return [];
        }

        $prefix = $currentYear->account_year_prefix;

        return [
            'assets_entry'                  => $prefix . '_assets_entry',
            'asset_transaction_record'      => $prefix . '_asset_mapping_transaction',
            'donor_transaction_record'      => $prefix . '_donor_mapping_transaction',
            'journal_transactions'          => $prefix . '_journal_transactions',
            'liability_transaction_record'  => $prefix . '_liability_mapping_transaction',
            'vendor_transaction_record'     => $prefix . '_vendor_mapping_transaction',
            'contra_journal_trans_mapping'  => $prefix . '_contra_journal_trans_mapping',
        ];
    }

    public function cashAndBankSumation()
    {
        $getTbles    = self::getYearPrefixedTables();
        $currentYear = self::getCurrentAccountingYear();
        $user = Auth::user();

        $journal_transactions =  $getTbles['journal_transactions'];
        $accounting_year_id   =  $currentYear->accounting_year_id;


        // 1️⃣ Get Brought Forward Values
        $accountBf = DB::table('account_brought_forwards')
            ->where('financial_year_id', $accounting_year_id)
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->select('id', 'cash_in_hand', 'cash_at_bank')
            ->first();


        // If no record found, default to 0 to prevent errors
        $cashInHandStart = $accountBf->cash_in_hand ?? 0;
        $cashAtBankStart = $accountBf->cash_at_bank ?? 0;

        // 2️⃣ Get total income and expense transactions

        $incomeCash = DB::table($journal_transactions)
            ->where('payment_type_id', 1)
            ->where('transcation_type', 'income')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');

        $incomeBank = DB::table($journal_transactions)
            ->where('payment_type_id', '>', 1)
            ->where('transcation_type', 'income')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');

        $expenseCash = DB::table($journal_transactions)
            ->where('payment_type_id', 1)
            ->where('transcation_type', 'expense')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');

        $expenseBank = DB::table($journal_transactions)
            ->where('payment_type_id', '>', 1)
            ->where('transcation_type', 'expense')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->sum('transaction_amount');

        // 3️⃣ Calculate contra entry adjustments
        // Get all contra journal entries directly from journal_transactions table
        $contraJournals = DB::table($journal_transactions)
            ->where('transcation_type', 'contra')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->select('id', 'contra_reference_type', 'transaction_amount', 'organisation_id')
            ->get();

        $contraCashAdjustment = 0;
        $contraBankAdjustment = 0;

        foreach ($contraJournals as $contraJournal) {
            $amount = floatval($contraJournal->transaction_amount ?? 0);

            switch ($contraJournal->contra_reference_type) {
                case 'cash_deposite':
                    // Cash deposited to bank: decrease cash, increase bank
                    // The transaction_amount represents the amount moved from cash to bank
                    $contraCashAdjustment -= $amount;
                    $contraBankAdjustment += $amount;
                    break;

                case 'cash_withdraw':
                    // Cash withdrawn from bank: increase cash, decrease bank
                    // The transaction_amount represents the amount moved from bank to cash
                    $contraCashAdjustment += $amount;
                    $contraBankAdjustment -= $amount;
                    break;

                case 'bank_transfer':
                    // Bank to bank transfer: no effect on total cash or total bank
                    // (money just moves between banks, so net effect is zero)
                    // Total bank balance remains the same, just redistributed
                    break;
            }
        }

        // 4️⃣ Calculate total balances including contra entries
        $totalCashInHand = ($cashInHandStart + $incomeCash) - $expenseCash + $contraCashAdjustment;
        $totalCashAtBank = ($cashAtBankStart + $incomeBank) - $expenseBank + $contraBankAdjustment;

        // 5️⃣ Return the results
        return [
            'account_bf_id' => $accountBf->id ?? null,
            'account_bf_cash_in_hand_opening' => $cashInHandStart,
            'account_bf_cash_at_bank_opening' => $cashAtBankStart,
            'total_cash_in_hand' => $totalCashInHand,
            'total_cash_at_bank' => $totalCashAtBank,
            'contra_cash_adjustment' => $contraCashAdjustment,
            'contra_bank_adjustment' => $contraBankAdjustment,
        ];
    }

    public function getPayment_TypeName_Idwise($type_id)
    {
        if (!$type_id) {
            return null; // No ID provided
        }

        // Fetch a single active payment type record
        $payment_type = PaymentType::select('id', 'name')
            ->where('status', 'active')
            ->where('id', $type_id)
            ->first();

        // Return object or null if not found
        return $payment_type ?: null;
    }

    public function getActiveBanks()
    {
        $banks = Bank::select('id', 'name', 'short_code', 'ifsc_code', 'branch', 'account_number')
            ->where('is_active', true)
            ->orderBy('name', 'ASC')
            ->get();

        return $banks;
    }

    public function getHeads($flag = null)
    {
        $user = Auth::user();
        $heads = Head::select('id', 'name')->where(['status' => 'active', 'organisation_id' => $user->organisation_id]);
        // Apply condition only if $flag is present
        if (!empty($flag)) {
            $heads->where('account_type_id', $flag);
        }
        $all_heads = $heads->orderBy('sort_order', 'ASC')->get();
        return $all_heads;
    }
    public function getHeadsByGroup($headGroupName, $incSubs = true)
    {
        $orgId = Auth::user()->organisation_id;
        $heads = Head::select('heads.id', 'heads.name')
            ->join('head_groups', 'heads.head_group_id', '=', 'head_groups.id');

        if (!$incSubs) {
            $heads
                ->where('heads.name', '!=', 'Yearly Subscription')
                ->where('heads.name', '!=', 'Subscription')
                ->where('heads.name', '!=', 'Contingency Fund');
        }
        $heads = $heads->where('heads.status', 'active')
            ->where('heads.organisation_id', $orgId)
            ->where('head_groups.name', $headGroupName)
            ->orderBy('heads.sort_order', 'asc')
            ->get()->toArray();
        return $heads;
    }

    public function getPaymentTypes()
    {
        return PaymentType::where('status', 'active')->get();
    }

    public function calculateAssetBroughtForward()
    {
        $getTbles = self::getYearPrefixedTables();
        $currentYear = self::getCurrentAccountingYear();
        $user = Auth::user();

        if (!$currentYear) {
            return [
                'dr_total' => 0,
                'cr_total' => 0,
                'net_asset' => 0,
                'opening_balance' => 0,
            ];
        }

        $assetMappingTable = $getTbles['asset_transaction_record'];
        $accounting_year_id = $currentYear->accounting_year_id;

        // 1️⃣ Get Brought Forward Values (Opening Balance)
        $assetBf = DB::table('asset_brought_forwards')
            ->where('financial_year_id', $accounting_year_id)
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->select('id', 'total_asset_amount')
            ->first();

        $openingBalance = $assetBf->total_asset_amount ?? 0;

        // 2️⃣ Calculate DR (Debit) total from asset_mapping_transaction table
        // Join with journal_transactions to filter by organisation_id if needed
        $drQuery = DB::table($assetMappingTable . ' as amt')
            ->leftJoin($getTbles['journal_transactions'] . ' as jt', 'amt.journal_id', '=', 'jt.id')
            ->where('amt.dr_cr', 'dr');

        if ($user && $user->id !== 1) {
            $drQuery->where('jt.organisation_id', $user->organisation_id);
        }

        $drTotal = $drQuery->sum('amt.amount') ?? 0;

        // 3️⃣ Calculate CR (Credit) total from asset_mapping_transaction table
        $crQuery = DB::table($assetMappingTable . ' as amt')
            ->leftJoin($getTbles['journal_transactions'] . ' as jt', 'amt.journal_id', '=', 'jt.id')
            ->where('amt.dr_cr', 'cr');

        if ($user && $user->id !== 1) {
            $crQuery->where('jt.organisation_id', $user->organisation_id);
        }

        $crTotal = $crQuery->sum('amt.amount') ?? 0;

        // 4️⃣ Calculate net asset amount
        // Net Asset = Opening Balance + DR - CR
        $netAsset = $openingBalance + $crTotal - $drTotal;

        // 5️⃣ Return the results
        return [
            'asset_bf_id' => $assetBf->id ?? null,
            'opening_balance' => $openingBalance,
            'dr_total' => $drTotal,
            'cr_total' => $crTotal,
            'net_asset' => $netAsset,
        ];
    }
    public function getOrganisationName($id)
    {
        $org_name = Organisation::where('organisation_id', $id)
            ->where('status', 'active')
            ->value('organisation_name');
        return $org_name;
    }
    public function getAccountTypesByOrg($id){
        $accountTypes = AccountType::where([
            'status' => 'active',
            'organisation_id' => $id
        ])->get()->toArray();

        return $accountTypes;
    }
}
