<?php

namespace App\Http\Controllers;

use App\Models\LiabilityBroughtForward;
use App\Models\AccountingYear;
use App\Models\Organisation;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class LiabilityBroughtForwardController extends Controller
{
    protected $currentYear;
    protected $currentOrg;

    public function __construct()
    {
        $this->currentYear = AccountingYear::current();
        $this->currentOrg = Organisation::current();
    }

    public function index()
    {
        $today = Carbon::now();
        $year = $today->year;

        if ($today->month < 4) {
            $startYear = $year - 1;
            $endYear   = $year;
        } else {
            $startYear = $year;
            $endYear   = $year + 1;
        }

        $financialYearStart = Carbon::createFromDate($startYear, 4, 1)->toDateString();

        $recordExists = LiabilityBroughtForward::where('dated_as_on', $financialYearStart)->count();

        return view('liabilityBroughtForward.index', compact('recordExists'));
    }

    public function getLiabilityBroughtForwardDetails(Request $request)
    {
        $search = $request->input('search');
        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = LiabilityBroughtForward::with('financialYear')
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('total_liability_amount', 'like', "%{$search}%")
                        ->orWhereHas(
                            'financialYear',
                            fn($q) =>
                            $q->where('accounting_year_financial', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);

        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        $years = AccountingYear::where('status', 'active')->get();
        return view('liabilityBroughtForward.create', compact('years'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'accounting_year_id' => 'required',
            'total_liability_amount' => 'required|numeric',
            'transaction_date'   => 'required',
        ]);

        if (!$this->currentOrg || !$this->currentOrg->organisation_id) {
            return redirect()->back()
                ->withInput()
                ->withErrors(
                    ['organisation_id' => 'Please select a current organisation before adding liability brought forward.'],
                    'orgCheck'
                );
        }

        try {
            LiabilityBroughtForward::create([
                'financial_year_id' => $validated['accounting_year_id'],
                'total_liability_amount' => $validated['total_liability_amount'],
                'user_id'           => auth()->user()->id,
                'dated_as_on'       => $validated['transaction_date'],
                'organisation_id'   => $this->currentOrg->organisation_id,
            ]);

            return redirect()->route('liability.brought.forward.index')->with('success', 'Liability Brought Forward added successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'organisation_id' => 'Please select a current organisation before adding liability brought forward. From Accounting Years add section.'
                ]);
        }
    }

    public function edit(string $id)
    {
        $id = base64_decode($id);
        $liabilityBroughtForward = LiabilityBroughtForward::where('id', $id)->first();
        $years = AccountingYear::where('status', 'active')->get();

        return view('liabilityBroughtForward.edit', compact('liabilityBroughtForward', 'years'));
    }

    public function update(Request $request, string $id)
    {
        $validated = $request->validate([
            'accounting_year_id' => 'required',
            'total_liability_amount' => 'required|numeric',
            'transaction_date' => 'required',
        ]);

        $id = base64_decode($id);
        $liabilityBroughtForward = LiabilityBroughtForward::findOrFail($id);

        $liabilityBroughtForward->update([
            'financial_year_id' => $validated['accounting_year_id'] ?? $liabilityBroughtForward->financial_year_id,
            'total_liability_amount' => $validated['total_liability_amount'] ?? $liabilityBroughtForward->total_liability_amount,
            'user_id'   => auth()->user()->id ?? $liabilityBroughtForward->user_id,
            'dated_as_on' => $validated['transaction_date'] ?? $liabilityBroughtForward->dated_as_on
        ]);

        return redirect()->route('liability.brought.forward.index')->with('success', 'Liability Brought Forward updated successfully.');
    }

    public function destroy(string $id)
    {
        try {
            $id = base64_decode($id);
            $liabilityBroughtForward = LiabilityBroughtForward::findOrFail($id);
            $liabilityBroughtForward->delete();

            return redirect()->route('liability.brought.forward.index')->with('success', 'Liability Brought Forward deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to delete Liability Brought Forward.');
        }
    }
}
