<?php

namespace App\Http\Controllers;
use App\Models\AccountingYear;
use App\Models\AccountType;
use App\Models\Asset;
use App\Models\AssetRecord;
use App\Models\Expense;
use App\Models\ExpenseHead;
use App\Models\ExpenseHeadGroup;
use App\Models\Liability;
use App\Models\LiabilityRecord;
use App\Models\Menu;
use App\Models\Organisation;
use App\Models\PaymentType;
use App\Models\Purpose;
use App\Models\Vendor;
use App\Models\VendorRecord;
use App\Services\PermissionService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use \Illuminate\Support\Facades\Auth;
use App\Traits\CommonTrait;


class ExpenseController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    use CommonTrait;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::where('route', 'expense.index')->value('id');
        $this->currentYear = AccountingYear::current();

    }

    public function index()
    {

        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Expense Records.');
        }
        $organisations = Organisation::where('status', 'active')->get();

        return view('auth.transactions.expense.index', compact('organisations'));
    }
    public function getExpenseDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Expense Records.');
        }

        $search = $request->input('search');
        $organisation_id = $request->input('organisation_id');

        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'expense_id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = Expense::with('organisation', 'accountType', 'expenseHead', 'paymentType', 'purpose')
            ->where('financial_year_id', $this->currentYear->accounting_year_id)
            ->when($organisation_id, fn($q) => $q->where('organisation_id', $organisation_id))
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('transaction_reference', 'like', "%{$search}%")
                        ->orWhere('transaction_amount', 'like', "%{$search}%")
                        ->orWhere('transaction_date', 'like', "%{$search}%")
                        ->orWhere('transaction_narration', 'like', "%{$search}%")
                        ->orWhereHas(
                            'organisation',
                            fn($q) =>
                            $q->where('organisation_name', 'like', "%{$search}%")
                        )
                        ->orWhereHas(
                            'accountType',
                            fn($q) =>
                            $q->where('account_type_name', 'like', "%{$search}%")
                        )
                        ->orWhereHas(
                            'expenseHead',
                            fn($q) =>
                            $q->where('expense_head_name', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Expense Records.');
        }

        $currentYear = $this->currentYear->accounting_year_financial;

        // return $currentYearId;
        $purposes = Purpose::where('status', 'active')->where('purpose_for', 'expense')
            ->get(['id', 'name']);
        $paymentTypes = PaymentType::where('status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes = $this->getAccountType();
        $expenseHeads = ExpenseHead::with('group')->where('status', 'active')->get();
        $vendors = Vendor::where('vendor_status', 'active')->get();
        return view('auth.transactions.expense.create', compact(
            'organisations',
            'accountTypes',
            'expenseHeads',
            'currentYear',
            'purposes',
            'paymentTypes',
            'vendors'
        ));
    }


    public function store(Request $request)
    {

        DB::beginTransaction();
        try {
            // Upload files
            $filePaths = [];
            if ($request->hasFile('expense_files')) {
                foreach ($request->file('expense_files') as $file) {
                    $filename = time() . '_' . $file->getClientOriginalName();
                    $filePaths[] = $file->storeAs('expense_files', $filename, 'public');
                }
            }

            list($expense_head_ids, $expense_group_head) = array_map('trim', explode('#', $request->expense_head_id));
            $filePathString = $filePaths ? implode(',', $filePaths) : null;

            // Create Expense entry
            $expense = Expense::create([
                'organisation_id'       => $request->organisation_id,
                'financial_year_id'     => $this->currentYear->accounting_year_id,
                'account_type_id'       => $request->account_type_id,
                'expense_head'          => $expense_head_ids,
                'user_id'               => Auth::id(),
                'transaction_amount'    => $request->transaction_amount,
                'payment_type_id'       => $request->payment_type_id,
                'transaction_date'      => $request->transaction_date,
                'transaction_reference' => $request->transaction_reference,
                'purpose_id'            => $request->transaction_purpose_id,
                'transaction_narration' => $request->transaction_narration,
                'receipt_no'            => $request->receipt_no,
                'file_path'             => $filePathString,
                'expense_type'          => 'expense',
            ]);

            // Helper: calculate depreciation
            $calcDepreciation = function ($headId, $amount) {
                $head = ExpenseHead::where(['status' => 'active', 'expense_head_id' => $headId])->first();
                if (!$head) return 0;

                $rate = (float) str_replace('%', '', $head->depriciation_rate ?? 0);
                $time = ($head->depriciation_tenure == 12) ? 1 : 0.5;

                return str_contains($head->depriciation_rate, '%')
                    ? ($amount * $rate * $time) / 100
                    : $rate;
            };

            // Asset-related entries
            if ($expense_group_head == 3) {
                $deprAmount = $calcDepreciation($expense_head_ids, $request->transaction_amount);

                Asset::create([
                    'organisation_id'              => $request->organisation_id,
                    'account_type_id'              => $request->account_type_id,
                    'expense_head_id'              => $expense_head_ids,
                    'asset_purchase_cost'          => $request->transaction_amount,
                    'depriciation_calcutated_amount' => $deprAmount,
                    'asset_purchase_date'          => $request->transaction_date,
                    'financial_year_id'            => $this->currentYear->accounting_year_id,
                ]);

                $assetRecord = [
                    'journal_id'      => $expense->expense_id,
                    'income_expense'  => 'expense',
                    'dr_cr'           => 'cr',
                    'amount'          => $request->transaction_amount,
                ];

                $vendorRecord = [
                    'journal_id'      => $expense->expense_id,
                    'income_expense'  => 'expense',
                    'dr_cr'           => 'cr',
                ];

                if ($request->payee_type === null) {
                    AssetRecord::create(array_merge($assetRecord, ['asset_id' => $request->expense_head_ids]));
                    $expense->update(['expense_type' => 'asset']);
                } elseif ($request->payee_type === 'external') {
                    AssetRecord::create(array_merge($assetRecord, [
                        'asset_id'  => $expense_head_ids,
                        'vendor_id' => $request->payee_id,
                        'amount'    => $request->vendor_total_amount,
                    ]));

                    VendorRecord::create(array_merge($vendorRecord, [
                        'asset_id'      => $expense_head_ids,
                        'vendor_id'     => $request->payee_id,
                        'total_amount'  => $request->vendor_total_amount,
                        'paid_amount'   => $request->transaction_amount,
                    ]));

                    $expense->update([
                        'expense_type' => 'asset',
                        'pay_to'       => 'external',
                        'vendor_id'    => $request->payee_id,
                    ]);
                } elseif ($request->payee_type === 'internal') {
                    AssetRecord::create(array_merge($assetRecord, [
                        'asset_id'  => $expense_head_ids,
                        'vendor_id' => $request->payee_id,
                    ]));

                    VendorRecord::create(array_merge($vendorRecord, [
                        'asset_id'    => $expense_head_ids,
                        'vendor_id'   => $request->payee_id,
                        'paid_amount' => $request->transaction_amount,
                    ]));

                    $expense->update([
                        'expense_type' => 'asset',
                        'pay_to'       => 'internal',
                        'staff_id'     => $request->payee_id,
                    ]);
                }
            }

            // Regular vendor/staff expenses
            if ($expense_group_head != 3 && ($request->payee_type !== null)) {

                $vendorRecord = [
                    'journal_id'      => $expense->expense_id,
                    'income_expense'  => 'expense',
                    'dr_cr'           => 'cr',
                    'vendor_id'       => $request->payee_id,
                    'paid_amount'     => $request->transaction_amount,
                ];

                if ($request->payee_type === 'external') {
                    $vendorRecord['total_amount'] = $request->vendor_total_amount;
                    $expense->update([
                        'expense_type' => 'expense',
                        'pay_to'       => 'external',
                        'vendor_id'    => $request->payee_id,
                    ]);
                } elseif ($request->payee_type === 'internal') {
                    $expense->update([
                        'expense_type' => 'expense',
                        'pay_to'       => 'internal',
                        'staff_id'     => $request->payee_id,
                    ]);
                }

                VendorRecord::create($vendorRecord);
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Expense stored successfully.',
                'expense' => $expense
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to store expense.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function eachDetail($id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Expense Records.');
        }
        $id = base64_decode($id);
        $assets = Asset::all();
        $liabilities = Liability::where('liability_status', 'active')->get();
        $vendors = Vendor::where('vendor_status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes = AccountType::where('status', 'active')->get();
        $expenseHeads = ExpenseHead::where('status', 'active')->get();
        $expense = Expense::with('paymentType', 'purpose')->where('expense_id', $id)->first();
        // return $expense;
        $accounting_years = AccountingYear::where('status', 'active')->get();
        return view('auth.transactions.expense.detail', compact('vendors', 'assets', 'liabilities', 'organisations', 'accountTypes', 'expense', 'accounting_years', 'expenseHeads'));
    }

    public function edit(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Expense Records.');
        }

        $id = base64_decode($id);
        $assets = Asset::select('asset_id', 'asset_name')->get();
        $liabilities = Liability::select('liability_id', 'liability_name')->where('liability_status', 'active')->get();
        $vendors = Vendor::select('vendor_id', 'vendor_name')->where('vendor_status', 'active')->get();
        $vendorRecord = VendorRecord::where('journal_id', $id)->first();
        // return $vendorRecord;

        $expense = Expense::where('expense_id', $id)->first();
        $existingFiles = [];
        if ($expense && $expense->file_path) {
            $existingFiles = explode(',', $expense->file_path);
        }
        $currentYear = $this->currentYear->accounting_year_financial;
        $purposes = Purpose::where('status', 'active')->where('purpose_for', 'expense')
            ->get(['id', 'name']);
        $paymentTypes = PaymentType::where('status', 'active')->get();
        // return $currentYearId;
        $expenseGroups = ExpenseHeadGroup::where('status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes = AccountType::where('status', 'active')->get();
        $expenseHeads = ExpenseHead::where('status', 'active')->get();
        return view('auth.transactions.expense.edit', compact('expenseGroups', 'existingFiles', 'vendorRecord', 'vendors', 'assets', 'liabilities', 'organisations', 'accountTypes', 'expenseHeads', 'currentYear', 'expense', 'purposes', 'paymentTypes'));
    }

    public function update(Request $request, $id)
    {
        // dd($request->all(), $request->file('expense_files'));
        // return $request;
        DB::beginTransaction();

        try {
            $expense = Expense::findOrFail($id);
            $filePaths = [];

            if ($request->hasFile('expense_files')) {
                // dd($request->file('income_files'));
                foreach ($request->file('expense_files') as $file) {
                    $filename = time() . '_' . $file->getClientOriginalName();
                    $path = $file->storeAs('expense_files', $filename, 'public');
                    $filePaths[] = $path;
                }
            }
            $filePathString = !empty($filePaths) ? implode(',', $filePaths) : null;

            $expenseData = [
                'organisation_id'       => $request->organisation_id,
                'financial_year_id' => $this->currentYear->accounting_year_id,
                'account_type_id'       => $request->account_type_id ?? NULL,
                'expense_head'          => $request->expense_head_id,
              'user_id'                 => Auth::id(),
                'transaction_amount'    => $request->transaction_amount,
                'payment_type_id'       => $request->payment_type_id,
                'transaction_date'      => $request->transaction_date ?? null,
                'transaction_reference' => $request->transaction_reference ?? null,
                'purpose_id' => $request->transaction_purpose_id,
                'transaction_narration' => $request->transaction_narration ?? NULL,

                // default nullable
                'receipt_no' =>  $request->receipt_no ?? NULL,
                'file_path'       =>  $filePathString,
                'expense_type' => 'expense',
                'asset_id'     => null,
                'liability_id' => null,
                'pay_to'       => null,
                'staff_id'     => null,
                'vendor_id'    => null,
                // 'vendor_total_amount'   => null,

            ];

            $expense->update($expenseData);

            AssetRecord::where('journal_id', $expense->expense_id)->delete();
            LiabilityRecord::where('journal_id', $expense->expense_id)->delete();
            VendorRecord::where('journal_id', $expense->expense_id)->delete();


            // handle ]asset] and [asset+vendor]:
            if ($request->type === 'asset' && $request->asset_id && $request->payee_type == null) {

                AssetRecord::create([
                    'journal_id' => $expense->getKey(),
                    'income_expense' => 'expense',
                    'dr_cr'     => 'cr',
                    'asset_id' => $request->asset_id,
                    'amount' =>  $request->transaction_amount,
                ]);
                $expense->update([
                    'expense_type' => 'asset',
                    'asset_id'     => $request->asset_id,
                ]);
            }

            if ($request->type === 'asset' && $request->asset_id && $request->payee_type == 'external') {

                AssetRecord::create([
                    'journal_id' => $expense->getKey(),
                    'income_expense' => 'expense',
                    'dr_cr'     => 'cr',
                    'asset_id' => $request->asset_id,
                    'vendor_id' => $request->payee_id,
                    'amount' =>  $request->vendor_total_amount,
                ]);
                VendorRecord::create([
                    'journal_id'   => $expense->getKey(),
                    'income_expense' => 'expense',
                    'asset_id' => $request->asset_id,
                    'vendor_id'     => $request->payee_id,
                    'total_amount' => $request->vendor_total_amount,
                    'paid_amount'  => $request->transaction_amount,
                    'dr_cr'     => 'cr',
                ]);
                $expense->update([
                    'expense_type' => 'asset',
                    'asset_id'     => $request->asset_id,
                    'pay_to' => 'external',
                    'vendor_id' =>  $request->payee_id
                ]);
            }
            if ($request->type === 'asset' && $request->asset_id && $request->payee_type == 'internal') {

                AssetRecord::create([
                    'journal_id' => $expense->getKey(),
                    'income_expense' => 'expense',
                    'dr_cr'     => 'cr',
                    'asset_id' => $request->asset_id,
                    'vendor_id' => $request->payee_id,
                    'amount' =>  $request->vendor_total_amount,
                ]);
                VendorRecord::create([
                    'journal_id'   => $expense->getKey(),
                    'income_expense' => 'expense',
                    'asset_id' => $request->asset_id,
                    'vendor_id'     => $request->payee_id,
                    'paid_amount' => $request->transaction_amount,
                    'dr_cr'     => 'cr',
                ]);
                $expense->update([
                    'expense_type' => 'asset',
                    'asset_id'     => $request->asset_id,
                    'pay_to' => 'internal',
                    'staff_id' =>  $request->payee_id
                ]);
            }

            // handle [liability] and [liability+vendor]:
            if ($request->type === 'liability' && $request->liability_id && $request->payee_type == null) {

                LiabilityRecord::create([
                    'journal_id'   => $expense->getKey(),
                    'liability_id' => $request->liability_id,
                    'amount'       => $request->transaction_amount,
                    'income_expense' => 'expense',
                    'dr_cr'     => 'dr',
                ]);
                $expense->update([
                    'expense_type' => 'liability',
                    'liability_id'     => $request->liability_id,
                ]);
            }

            if ($request->type === 'liability' && $request->liability_id && $request->payee_type == 'external') {
                LiabilityRecord::create([
                    'journal_id' => $expense->getKey(),
                    'income_expense' => 'expense',
                    'dr_cr'     => 'cr',
                    'liability_id' => $request->liability_id,
                    'vendor_id' => $request->payee_id,
                    'amount' =>  $request->transaction_amount,
                ]);
                VendorRecord::create([
                    'journal_id'   => $expense->getKey(),
                    'income_expense' => 'expense',
                    'liability_id' => $request->liability_id,
                    'vendor_id'     => $request->payee_id,
                    'total_amount' => $request->vendor_total_amount,
                    'paid_amount'  => $request->transaction_amount,
                    'dr_cr'     => 'dr',
                ]);
                $expense->update([
                    'expense_type' => 'liability',
                    'liability_id'     => $request->liability_id,
                    'pay_to' => 'external',
                    'vendor_id' =>  $request->payee_id
                ]);
            }

            if ($request->type === 'liability' && $request->liability_id && $request->payee_type == 'internal') {
                LiabilityRecord::create([
                    'journal_id' => $expense->getKey(),
                    'income_expense' => 'expense',
                    'dr_cr'     => 'cr',
                    'liability_id' => $request->liability_id,
                    'vendor_id' => $request->payee_id,
                    'amount' =>  $request->transaction_amount,
                ]);
                VendorRecord::create([
                    'journal_id'   => $expense->getKey(),
                    'income_expense' => 'expense',
                    'liability_id' => $request->liability_id,
                    'vendor_id'     => $request->payee_id,
                    'paid_amount'  => $request->transaction_amount,
                    'dr_cr'     => 'dr',
                ]);
                $expense->update([
                    'expense_type' => 'liability',
                    'liability_id'     => $request->liability_id,
                    'pay_to' => 'internal',
                    'staff_id' =>  $request->payee_id
                ]);
            }

            //for only staff and only external vendor:
            if ($request->payee_type == 'internal' && $request->type == null) {

                VendorRecord::create([
                    'journal_id'   => $expense->getKey(),
                    'income_expense' => 'expense',
                    'vendor_id'     => $request->payee_id,
                    'paid_amount'  => $request->transaction_amount,
                    'dr_cr'     => 'cr',
                ]);
                $expense->update([
                    'expense_type' => 'expense',
                    'pay_to' => 'internal',
                    'staff_id' =>  $request->payee_id
                ]);
            }


            if ($request->payee_type == 'external' && $request->type == null) {
                //entry balanced
                VendorRecord::create([
                    'journal_id'   => $expense->getKey(),
                    'income_expense' => 'expense',
                    'vendor_id'     => $request->payee_id,
                    'total_amount' => $request->vendor_total_amount,
                    'paid_amount'  => $request->transaction_amount,
                    'dr_cr'     => 'cr',
                ]);
                $expense->update([
                    'expense_type' => 'expense',
                    'pay_to' => 'external',
                    'vendor_id' =>  $request->payee_id
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message'  => 'Expense updated successfully!',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to store expense',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function destroy(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Expense Records.');
        }
        // return $id;
        $expense = Expense::findOrFail($id);
        DB::table('vendor_records')
            ->where('journal_id', $expense->expense_id)
            ->delete();
        $expense->delete();

        return response()->json([
            'status'  => true,
            'message' => 'Expense deleted successfully.'
        ]);
    }

    public function print($id)
    {
        $expense = Expense::with(['paymentType'])->findOrFail($id);

        $organisations = Organisation::all();
        $accountTypes = AccountType::all();
        $expenseHeads  = ExpenseHead::all();
        $assets       = Asset::all();
        $liabilities  = Liability::all();
        $vendors       = Vendor::all();
        $accounting_years = AccountingYear::all();

        $pdf = Pdf::loadView('auth.transactions.expense.print', compact(
            'expense',
            'organisations',
            'accountTypes',
            'expenseHeads',
            'assets',
            'liabilities',
            'vendors',
            'accounting_years'
        ));

        return $pdf->stream('expense-transaction-' . $expense->expense_id . '.pdf');
    }
}
