<?php

namespace App\Http\Controllers;

use App\Models\AccountBroughtForward;
use App\Models\AccountingYear;
use App\Models\AccountType;
use App\Models\Expense;
use App\Models\Income;
use App\Models\Menu;
use App\Models\Organisation;
use App\Services\PermissionService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CapitalController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    protected $currentOrg;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;

        $this->menuId = Menu::where('route', 'capital.index')->value('id');
        $this->currentYear = AccountingYear::current();
        $this->currentOrg = Organisation::current();
    }

    public function index()
    {

        $today = Carbon::now();
        $year = $today->year;

        if ($today->month < 4) {
            $startYear = $year - 1;
            $endYear   = $year;
        } else {
            $startYear = $year;
            $endYear   = $year + 1;
        }

        $financialYearStart = Carbon::createFromDate($startYear, 4, 1)->toDateString();

        // return $financialYearStart;      
        $recordExists = AccountBroughtForward::where('dated_as_on', $financialYearStart)->count();
        // return $recordExists;
        return view('auth.capitals.index', compact('recordExists'));
    }



    public function getcapitalDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Capital Records.');
        }

        $search = $request->input('search');
        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = AccountBroughtForward::with('financialYear')
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('cash_in_hand', 'like', "%{$search}%")
                        ->orWhere('cash_at_bank', 'like', "%{$search}%")
                        ->orWhereHas(
                            'financialYear',
                            fn($q) =>
                            $q->where('accounting_year_financial', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        $years = AccountingYear::where('status', 'active')->get();
        return view('auth.capitals.create', compact('years'));
    }

    public function store(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Capital Records.');
        }

        $validated = $request->validate([
            'accounting_year_id' => 'required',
            'cash_in_hand'       => 'required',
            'cash_at_bank'       => 'required',
            'transaction_date'   => 'required',
        ]);
        if (!$this->currentOrg || !$this->currentOrg->organisation_id) {
            return redirect()->back()
                ->withInput()
                ->withErrors(
                    ['organisation_id' => 'Please select a current organisation before adding user.'],
                    'orgCheck'
                );
        }
        try {
            AccountBroughtForward::create([
                'financial_year_id' => $validated['accounting_year_id'],
                'cash_in_hand'      => $validated['cash_in_hand'],
                'cash_at_bank'      => $validated['cash_at_bank'],
                'user_id'           => auth()->user()->id,
                'dated_as_on'       => $validated['transaction_date'],
                'organisation_id'   => $this->currentOrg->organisation_id,
            ]);

            return redirect()->route('capital.index')->with('success', 'Capital added successfully.');
        } catch (QueryException $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'organisation_id' => 'Please select a current organisation before adding capital. From Accounting Years add section.'
                ]);
        }
    }



    public function show(string $id)
    {
        //
    }


    public function edit(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Capital Records.');
        }

        $id = base64_decode($id);
        // return $id;
        $accountCapital = AccountBroughtForward::where('id', $id)->first();
        $years = AccountingYear::where('status', 'active')->get();

        return view('auth.capitals.edit', compact('accountCapital', 'years'));
    }


    public function update(Request $request, string $id)
    {
        // return $request;
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have write access to Capital Records.');
        }

        $validated = $request->validate([
            'accounting_year_id' => 'required',
            'cash_in_hand'          => 'required',
            'cash_at_bank'  => 'required',
            'transaction_date' => 'required',
        ]);

        $accountCapital = AccountBroughtForward::findOrFail($id);

        $accountCapital->update([
            'financial_year_id' => $validated['accounting_year_id'] ?? $accountCapital->financial_year_id,
            'cash_in_hand' => $validated['cash_in_hand'] ?? $accountCapital->cash_in_hand,
            'cash_at_bank'    => $validated['cash_at_bank'] ?? $accountCapital->cash_at_bank,
            'user_id'   => auth()->user()->id ?? $accountCapital->user_id,
            'dated_as_on' => $validated['transaction_date'] ?? $accountCapital->dated_as_on
        ]);
        return redirect()->route('capital.index')->with('success', 'Capital updated successfully.');
    }

    public function destroy(string $id)
    {
        //
    }

    public function netCapital()
    {
        $id = $this->currentYear->accounting_year_id;

        $capital = AccountBroughtForward::where('financial_year_id', $id)->first();

        $incomeCash = DB::table('incomes')->whereNull('deleted_at')->where('payment_type_id', 1)->sum('transaction_amount');
        $incomeBank = DB::table('incomes')->whereNull('deleted_at')->where('payment_type_id', '>', 1)->sum('transaction_amount');

        $expenseCash = DB::table('expenses')->whereNull('deleted_at')->where('payment_type_id', 1)->sum('transaction_amount');
        $expenseBank = DB::table('expenses')->whereNull('deleted_at')->where('payment_type_id', '>', 1)->sum('transaction_amount');

        $totalCash = ($capital->cash_in_hand ?? 0) +  $incomeCash - $expenseCash;
        $totalBank = ($capital->cash_at_bank ?? 0) + $incomeBank - $expenseBank;

        return view('auth.capitals.total', compact('totalCash', 'totalBank'));
    }
    public function getcapitalAmountDetails($flag)
    {
        $id = $this->currentYear->accounting_year_id;

        $capital = AccountBroughtForward::where('financial_year_id', $id)->first();

        if ($flag === 'cash') {
            $incomeDatas = Income::with(['incomeHead', 'donor'])
                ->whereNull('deleted_at')
                ->where('payment_type_id', 1)
                ->orderByDesc('transaction_date')
                ->get();

            $expenseDatas = Expense::with(['expenseHead', 'vendor', 'staffVendor'])
                ->whereNull('deleted_at')
                ->where('payment_type_id', 1)
                ->orderByDesc('transaction_date')
                ->get();
            $forwardCapital = $capital->cash_in_hand ?? 0;
        } elseif ($flag === 'bank') {
            $incomeDatas = Income::with(['incomeHead', 'donor'])
                ->whereNull('deleted_at')
                ->where('payment_type_id', '>', 1)
                ->orderByDesc('transaction_date')
                ->get();

            $expenseDatas = Expense::with(['expenseHead', 'vendor', 'staffVendor'])
                ->whereNull('deleted_at')
                ->where('payment_type_id', '>', 1)
                ->orderByDesc('transaction_date')
                ->get();
            $forwardCapital = $capital->cash_at_bank ?? 0;
        } else {
            abort(404, 'Invalid flag');
        }

        // return $expenseDatas;
        return view('auth.capitals.totalHistory', compact('incomeDatas', 'expenseDatas', 'flag', 'forwardCapital'));
    }
}
