<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Carbon\Carbon;
use DB;
use Illuminate\Support\Str;


use App\Models\SchoolClass;
use App\Models\SchoolSection;
use App\Models\Student;
use App\Models\AcademicYear;
use App\Models\ClassFeeConfig;
use App\Models\StudentMonthlyFee;
use App\Models\TransactionIncome;

class StudentFeeController extends Controller
{
    public function index(Request $request)
    {
        /* -------------------------------------------------
            Filters
        ------------------------------------------------- */
        $financialYearId = $request->financial_year_id;
        $classId         = $request->class_id;
        $sectionId       = $request->section_id;
        $month           = $request->filled('month')
                            ? $request->month
                            : now()->format('Y-m');
        $search          = $request->q;

        /* -------------------------------------------------
            Masters
        ------------------------------------------------- */
        $academic_years = AcademicYear::where('status', 1)->get();

        $active_academic_year = AcademicYear::where('status', 1)->firstOrFail();

        // Always default to active academic year
        $financialYearId = $financialYearId ?: $active_academic_year->id;

        $selectedAcademicYear = AcademicYear::findOrFail($financialYearId);

        // Academic year month range
        $yearStartMonth = \Carbon\Carbon::parse($selectedAcademicYear->year_from)->format('Y-m');
        $yearEndMonth   = \Carbon\Carbon::parse($selectedAcademicYear->year_to)->format('Y-m');

        $classes = SchoolClass::where('status', 1)->get();

        $sections = collect();
        if ($classId && $classId !== 'all') {
            $sections = SchoolSection::where('class_id', $classId)->get();
        }

        /* -------------------------------------------------
            ❌ Month outside academic year → empty students
        ------------------------------------------------- */
        if ($month < $yearStartMonth || $month > $yearEndMonth) {

            $students = Student::whereRaw('1 = 0')
                ->paginate(25)
                ->withQueryString();

            return view('student_fees.index', compact(
                'academic_years',
                'active_academic_year',
                'classes',
                'sections',
                'students'
            ));
        }

        /* -------------------------------------------------
            ❌ NO FEE CONFIG → NO STUDENTS
        ------------------------------------------------- */
        $hasFeeConfig = ClassFeeConfig::where('financial_year_id', $financialYearId)
            ->where('status', 1)
            ->when($classId && $classId !== 'all', function ($q) use ($classId) {
                $q->where('class_id', $classId);
            })
            ->exists();

        if (!$hasFeeConfig) {

            $students = Student::whereRaw('1 = 0')
                ->paginate(25)
                ->withQueryString();

            return view('student_fees.index', compact(
                'academic_years',
                'active_academic_year',
                'classes',
                'sections',
                'students'
            ));
        }

        /* -------------------------------------------------
            Students Query
        ------------------------------------------------- */
        $students = Student::with([
                'activeClass.class',
                'activeClass.section'
            ])
            ->where('status', 1)

            // Restrict by academic year
            ->whereHas('activeClass', function ($q) use ($financialYearId) {
                $q->where('financial_year_id', $financialYearId)
                ->where('status', 1);
            })

            // Filter by class
            ->when($classId && $classId !== 'all', function ($q) use ($classId) {
                $q->whereHas('activeClass', function ($qq) use ($classId) {
                    $qq->where('class_id', $classId);
                });
            })

            // Filter by section
            ->when($sectionId, function ($q) use ($sectionId) {
                $q->whereHas('activeClass', function ($qq) use ($sectionId) {
                    $qq->where('section_id', $sectionId);
                });
            })

            // Search
            ->when($search, function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('name', 'like', "%{$search}%")
                    ->orWhere('admission_no', 'like', "%{$search}%");
                });
            })

            ->orderBy('name')
            ->paginate(25)
            ->withQueryString();

        /* -------------------------------------------------
            Fee Calculation
        ------------------------------------------------- */
        foreach ($students as $student) {

            $activeClass = $student->activeClass;

            if (!$activeClass) {
                $student->total_fee  = 0;
                $student->fee_status = 'N/A';
                continue;
            }

            $configs = ClassFeeConfig::where([
                    'class_id' => $activeClass->class_id,
                    'financial_year_id' => $financialYearId,
                    'status' => 1,
                ])->get();

            if ($configs->isEmpty()) {
                $student->total_fee  = 0;
                $student->fee_status = 'N/A';
                continue;
            }

            $totalDue    = 0;
            $isFullyPaid = true;

            foreach ($configs as $cfg) {

                $isApplicable = false;

                // Recurring fees
                if ($cfg->fee_type === 'recurring') {
                    $isApplicable = true;
                }

                // One-time fees
                if (
                    $cfg->fee_type === 'one_time' &&
                    $cfg->due_date &&
                    \Carbon\Carbon::parse($cfg->due_date)->format('Y-m') === $month
                ) {
                    $isApplicable = true;
                }

                if (!$isApplicable) {
                    continue;
                }

                $paid = StudentMonthlyFee::where([
                    'student_id'        => $student->id,
                    'fee_head_id'       => $cfg->fee_head_id,
                    'financial_year_id' => $financialYearId,
                    'fee_for_month'     => $month,
                ])->exists();

                if (!$paid) {
                    $isFullyPaid = false;

                    $lateFee = 0;

                    if ($cfg->fee_type === 'recurring' && $cfg->due_day_of_month) {
                        if (now()->day > $cfg->due_day_of_month) {
                            $lateFee = $cfg->late_fee_amount ?? 0;
                        }
                    }

                    if ($cfg->fee_type === 'one_time' && $cfg->due_date) {
                        if (now()->gt(\Carbon\Carbon::parse($cfg->due_date))) {
                            $lateFee = $cfg->late_fee_amount ?? 0;
                        }
                    }

                    $totalDue += ($cfg->amount + $lateFee);
                }
            }

            $student->total_fee  = $totalDue;
            $student->fee_status = $isFullyPaid ? 'paid' : 'unpaid';
        }

        return view('student_fees.index', compact(
            'academic_years',
            'active_academic_year',
            'classes',
            'sections',
            'students'
        ));
    }





    /* =========================================================
        FETCH STUDENT FEES (AJAX)
    ========================================================= */
   public function getStudentFees(Student $student, Request $request)
    {
        $academicYear = AcademicYear::where('status', 1)->firstOrFail();
        $yearId = $academicYear->id;

        $classId = optional($student->activeClass)->class_id;
        if (!$classId) {
            return response()->json([]);
        }

        $configs = ClassFeeConfig::with('feeHead')
            ->where('class_id', $classId)
            ->where('financial_year_id', $yearId)
            ->where('status', 1)
            ->get();

        // ❌ No fee head → student should not appear
        if ($configs->isEmpty()) {
            return response()->json([]);
        }

        $start = Carbon::parse($academicYear->year_from)->startOfMonth();
        $end   = Carbon::parse($academicYear->year_to)->startOfMonth();

        $months = [];

        while ($start <= $end) {

            $monthKey = $start->format('Y-m');

            $months[$monthKey] = [
                'month'      => $monthKey,
                'label'      => $start->format('F Y'),
                'fees'       => [],
                'total'      => 0,
                'status'     => '',
                'selectable' => false
            ];

            $hasFee      = false;
            $hasUnpaid   = false;

            foreach ($configs as $cfg) {

                /* ===============================
                    RECURRING FEES
                =============================== */
                if ($cfg->fee_type === 'recurring') {

                    $hasFee = true;

                    $paid = StudentMonthlyFee::where([
                        'student_id'         => $student->id,
                        'fee_head_id'        => $cfg->fee_head_id,
                        'financial_year_id' => $yearId,
                        'fee_for_month'     => $monthKey,
                    ])->exists();

                    $lateFee = 0;

                    if (!$paid && $cfg->due_day_of_month) {
                        $dueDate = Carbon::createFromFormat(
                            'Y-m-d',
                            $monthKey . '-' . str_pad($cfg->due_day_of_month, 2, '0', STR_PAD_LEFT)
                        );

                        if (now()->gt($dueDate)) {
                            $lateFee = $cfg->late_fee_amount ?? 0;
                        }
                    }

                    $amountWithLate = $cfg->amount + $lateFee;
                    $months[$monthKey]['total'] += $amountWithLate;

                    $months[$monthKey]['fees'][] = [
                        'fee_head_id' => $cfg->fee_head_id,
                        'name'        => $cfg->feeHead->name,
                        'type'        => 'recurring',
                        'amount'      => $cfg->amount,
                        'late_fee'    => $lateFee,
                        'total'       => $amountWithLate,
                        'paid'        => $paid
                    ];

                    if (!$paid) {
                        $hasUnpaid = true;
                    }
                }

                /* ===============================
                    ONE-TIME FEES
                =============================== */
                if ($cfg->fee_type === 'one_time' && $cfg->due_date) {

                    if (Carbon::parse($cfg->due_date)->format('Y-m') !== $monthKey) {
                        continue;
                    }

                    $hasFee = true;

                    $paid = StudentMonthlyFee::where([
                        'student_id'         => $student->id,
                        'fee_head_id'        => $cfg->fee_head_id,
                        'financial_year_id' => $yearId,
                    ])->exists();

                    $lateFee = (!$paid && now()->gt(Carbon::parse($cfg->due_date)))
                        ? ($cfg->late_fee_amount ?? 0)
                        : 0;

                    $amountWithLate = $cfg->amount + $lateFee;
                    $months[$monthKey]['total'] += $amountWithLate;

                    $months[$monthKey]['fees'][] = [
                        'fee_head_id' => $cfg->fee_head_id,
                        'name'        => $cfg->feeHead->name,
                        'type'        => 'one_time',
                        'amount'      => $cfg->amount,
                        'late_fee'    => $lateFee,
                        'total'       => $amountWithLate,
                        'paid'        => $paid
                    ];

                    if (!$paid) {
                        $hasUnpaid = true;
                    }
                }
            }

            /* ===============================
                FINAL STATUS LOGIC (FIX)
            =============================== */
            if ($hasFee) {
                if ($hasUnpaid) {
                    $months[$monthKey]['status']     = 'unpaid';
                    $months[$monthKey]['selectable'] = true;
                } else {
                    $months[$monthKey]['status']     = 'paid';
                    $months[$monthKey]['selectable'] = false;
                }
            }

            $start->addMonth();
        }

        return response()->json(array_values($months));
    }






    /* =========================================================
        STORE PAYMENT
    ========================================================= */
    public function storePayment(Request $request)
    {
        // $request->validate([
        //     'student_id'         => 'required|exists:students,id',
        //     'financial_year_id'  => 'required|exists:academic_years,id',
        //     'payment_method'     => 'required|string',
        //     'transaction_id'     => 'nullable|string',
        //     'months'             => 'required|array',
        //     'fees'               => 'required|array',
        // ]);

        // echo '<pre>';
        // print_r($request->toArray());die;

        DB::beginTransaction();

        try {
            
            $total = 0;
            foreach ($request->months as $month) {

                if (!isset($request->fees[$month])) {
                    continue;
                }

                
                foreach ($request->fees[$month] as $fee) {

                    // 🔒 Prevent duplicate charge
                    $alreadyPaid = StudentMonthlyFee::where([
                        'student_id'         => $request->student_id,
                        'fee_head_id'        => $fee['fee_head_id'],
                        'financial_year_id' => $request->financial_year_id,
                        'fee_for_month'     => $month,
                    ])->exists();

                    if ($alreadyPaid) {
                        continue;
                    }

                    StudentMonthlyFee::create([
                        'student_id'        => $request->student_id,
                        'fee_head_id'       => $fee['fee_head_id'],
                        'financial_year_id' => $request->financial_year_id,
                        'fee_for_month'     => $month,
                        'amount'            => $fee['amount'],
                        'late_fee'          => $fee['late_fee'] ?? 0,
                        'paid_date'         => now(),
                        'payment_method'    => $request->payment_method,
                        'transaction_id'    => $request->transaction_id,
                    ]);

                    $total = $total+$fee['amount']+($fee['late_fee'] ?? 0);

                }
            }

            if(count($request->fees[$month]) > 0 && $total>0) {

                TransactionIncome::create([
                    'transaction_id' => strtoupper(Str::random(10)),
                    'transaction_ref_id' => $request->transaction_id,
                    'financial_year_id' => $request->financial_year_id,
                    'transaction_amount' => $total,
                    'transaction_date' => now(),
                    'payment_method' => $request->payment_method,
                    'income_head_id' => 1,
                    'source_type' => 'student',
                    'source_id' => $request->student_id,
                ]);
            }

            DB::commit();

            // return response()->json([
            //     'success' => true,
            //     'message' => 'Fee collected successfully'
            // ]);

            return back()->with('success', 'Fee collected successfully.');

        } catch (\Throwable $e) {

            DB::rollBack();

            // return response()->json([
            //     'success' => false,
            //     'message' => 'Failed to collect fee',
            //     'error'   => $e->getMessage()
            // ], 500);
            return back()->with('error', 'The student has no outstanding monthly fees.');
        }
    }

    public function monthDetails(Student $student, Request $request)
    {
        $month = $request->month;

        $student->load([
            'activeClass.class',
            'activeClass.section'
        ]);

        $fees = StudentMonthlyFee::with('feeHead')
            ->where('student_id', $student->id)
            ->where('fee_for_month', $month)
            ->get();

        return response()->json([
            'student' => [
                'name'          => $student->name,
                'admission_no'  => $student->admission_no,
                'class'         => optional($student->activeClass->class)->class_name,
                'section'       => optional($student->activeClass->section)->section_name,
                'roll_no'       => optional($student->activeClass)->roll_no,
            ],
            'month' => $month,
            'fees'  => $fees
        ]);
    }


    public function monthPendingFees(Student $student, Request $request)
    {
        $month = $request->month;
        $yearId = $request->financial_year_id;

        $classId = optional($student->activeClass)->class_id;
        if (!$classId) return response()->json([]);

        $configs = ClassFeeConfig::with('feeHead')
            ->where('class_id', $classId)
            ->where('financial_year_id', $yearId)
            ->where('status', 1)
            ->get();

        $fees = [];

        foreach ($configs as $cfg) {

            $alreadyPaid = StudentMonthlyFee::where([
                'student_id' => $student->id,
                'fee_head_id'=> $cfg->fee_head_id,
                'financial_year_id'=> $yearId,
                'fee_for_month'=> $month
            ])->exists();

            if ($alreadyPaid) continue;

            $late = 0;

            if ($cfg->fee_type === 'recurring' && $cfg->due_day_of_month) {
                if (now()->day > $cfg->due_day_of_month) {
                    $late = $cfg->late_fee_amount ?? 0;
                }
            }

            if ($cfg->fee_type === 'one_time' &&
                $cfg->due_date &&
                Carbon::parse($cfg->due_date)->format('Y-m') === $month
            )  {
                if (now()->gt($cfg->due_date)) {
                    $late = $cfg->late_fee_amount ?? 0;
                }
            }

            $fees[] = [
                'fee_head_id' => $cfg->fee_head_id,
                'name' => $cfg->feeHead->name,
                'amount' => $cfg->amount,
                'late_fee' => $late,
                'total' => $cfg->amount + $late
            ];
        }

        return response()->json([
            'month' => $month,
            'fees' => $fees
        ]);
    }

    public function paySingleMonth(Request $request)
    {
        $request->validate([
            'student_id'        => 'required|exists:students,id',
            'financial_year_id' => 'required|exists:financial_years,id',
            'fee_for_month'     => 'required',
            'payment_method'    => 'required',
            'transaction_id'    => 'required',
            'fees'              => 'required|array|min:1',
        ]);

        DB::transaction(function () use ($request) {

            $total = 0;
            foreach ($request->fees as $fee) {

                // 🛑 Prevent duplicate payment
                $exists = StudentMonthlyFee::where([
                    'student_id'        => $request->student_id,
                    'fee_head_id'       => $fee['fee_head_id'],
                    'financial_year_id' => $request->financial_year_id,
                    'fee_for_month'     => $request->fee_for_month
                ])->exists();

                if ($exists) {
                    continue;
                }

                StudentMonthlyFee::create([
                    'student_id'        => $request->student_id,
                    'fee_head_id'       => $fee['fee_head_id'],
                    'financial_year_id' => $request->financial_year_id,
                    'fee_for_month'     => $request->fee_for_month,

                    'amount'            => $fee['amount'],
                    'late_fee'          => $fee['late_fee'] ?? 0,

                    'paid_date'         => now(),
                    'payment_method'    => $request->payment_method,
                    'transaction_id'    => $request->transaction_id,
                ]);
                $total = $total + $fee['amount'] +($fee['late_fee'] ?? 0) ;
            }

            if(count($request->fees) > 0 && $total>0) {

                TransactionIncome::create([
                    'transaction_id' => strtoupper(Str::random(10)),
                    'transaction_ref_id' => $request->transaction_id,
                    'financial_year_id' => $request->financial_year_id,
                    'transaction_amount' => $total,
                    'transaction_date' => now(),
                    'payment_method' => $request->payment_method,
                    'income_head_id' => 1,
                    'source_type' => 'student',
                    'source_id' => $request->student_id,
                ]);
            }
        });

        return back()->with('success', 'Fee collected successfully.');
    }

}
