<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Throwable;
use PhpOffice\PhpSpreadsheet\Shared\Date as ExcelDate;
use Carbon\Carbon;
use App\Models\Student;
use App\Models\ParentModel;
use App\Models\StudentClassDetail;
use App\Exports\StudentTemplateExport;


class StudentBulkUploadController extends Controller
{
    protected array $headerMap;

    /* =====================================================
     | UPLOAD ENTRY POINT
     ===================================================== */
    public function upload(Request $request)
    {
        //session(['tab' => 'bluk']);
        $request->validate([
            'file'               => 'required|mimes:csv,xlsx',
            'class_id'           => 'required|exists:classes,id',
            'section_id'         => 'required|exists:sections,id',
            'financial_year_id'  => 'required|exists:financial_years,id',
        ]);

        $rows = Excel::toArray([], $request->file('file'))[0] ?? [];

        // echo '<pre>';
        // print_r($rows);die;
        if (empty($rows) || empty($rows[0])) {
            return back()->withErrors(['file' => 'Uploaded file is empty']);
        }

        $this->headerMap = array_values($this->bulkHeaderMap());

        if (array_map('trim', $rows[0]) !== array_keys($this->bulkHeaderMap())) {
            return back()->withErrors(['file' => 'Uploaded file format does not match template']);
        }

        /* Count valid rows */
        $totalRows = 0;
        foreach ($rows as $index => $row) {
            if ($index === 0) continue;
            if (!$this->isBlankRow($row)) $totalRows++;
        }

        $path = $request->file('file')->store('student_bulk', 'local');

        $uploadId = DB::table('student_bulk_uploads')->insertGetId([
            'total_rows'     => $totalRows,
            'processed_rows' => 0,
            'status'         => 'completed',
            'created_at'     => now(),
            'updated_at'     => now(),
        ]);
        //echo $uploadId;

        $this->bulkUpload(
            $path,
            $uploadId,
            $request->class_id,
            $request->section_id,
            $request->financial_year_id
        );

        return back()->with(['tab' => 'bulk'])->with('success', 'Bulk upload finished. Please check history.');
    }

    /* =====================================================
     | CORE BULK PROCESS (NO JOB)
     ===================================================== */
    private function bulkUpload(
        string $path,
        int $uploadId,
        int $classId,
        int $sectionId,
        int $financialYearId
    ): void {
        $hasErrors = false;
        
        DB::table('student_bulk_uploads')
            ->where('id', $uploadId)
            ->update(['status' => 'pending']);

        //echo $uploadId; die;
        try {
            $rows = Excel::toArray([], Storage::disk('local')->path($path))[0] ?? [];
            unset($rows[0]); // remove header

            foreach ($rows as $index => $row) {

                if ($this->isBlankRow($row)) {
                    continue;
                }

                $rowNumber = $index + 1;
                $data = $this->mapRow($row);

                /* Phone split */
                $phones = explode('/', (string) $data['father_phone']);
                
                $data['father_phone'] = trim($phones[0] ?? null);
                $data['mother_phone'] = trim($phones[1] ?? null);
                $data['guardian_phone'] = trim($phones[2] ?? null);

                /* Date conversion */
                //$data['dob'] = $this->excelDateToDate($data['dob']);
                //$data['admission_date'] = $this->excelDateToDate($data['admission_date']);

                /* Validation */
                $validator = $this->validator($data);
                if ($validator->fails()) {
                    $this->failRow($uploadId, $rowNumber, $data, $validator->errors()->first());
                    $hasErrors = true;
                    $this->increment($uploadId);
                    continue;
                }

                /* Duplicate check */
                if (
                    Student::where('name', $data['name'])
                        ->whereHas('parent', fn ($q) =>
                            $q->where('father_phone', $data['father_phone'])
                        )->exists()
                ) {
                    $this->failRow($uploadId, $rowNumber, $data, 'Duplicate student');
                    $hasErrors = true;
                    $this->increment($uploadId);
                    continue;
                }

                /*$student = Student::where('name', $data['name'])
                    ->whereHas('parent', function ($q) use ($data) {
                        $q->where('father_phone', $data['father_phone']);
                    })
                    ->first();

                if ($student) {

                    // 🔒 Check duplicate in SAME class & financial year
                    $existingClass = StudentClassDetail::where([
                        'student_id'        => $student->id,
                        'class_id'          => $classId,
                        'financial_year_id' => $financialYearId,
                    ])->first();

                    if ($existingClass) {
                        $this->failRow($uploadId, $rowNumber, $data, 'Duplicate student for this class & year');
                        $hasErrors = true;
                        $this->increment($uploadId);
                        continue;
                    }

                    // 🔄 Deactivate previous active class records
                    StudentClassDetail::where('student_id', $student->id)
                        ->where('financial_year_id', $financialYearId)
                        ->where('status', 1)
                        ->update(['status' => 0,'end_date' => now()]);

                    // ➕ Assign new class
                    StudentClassDetail::create([
                        'student_id'        => $student->id,
                        'class_id'          => $classId,
                        'section_id'        => $sectionId,
                        'roll_no'           => $roll ? $roll + 1 : 1,
                        'financial_year_id' => $financialYearId,
                        'admission_date'    => Carbon\Carbon::parse($data['admission_date'])->format('Y-m-d'),
                        'start_date'        => now(),
                        'status'            => 1,
                    ]);
                }*/


                try {
                    DB::transaction(function () use ($data, $classId, $sectionId, $financialYearId) {


                        $parent = ParentModel::create([
                            'father_name'  => $data['father_name'],
                            'mother_name'  => $data['mother_name'],
                            'father_phone' => str_replace('+91','',$data['father_phone']),
                            'mother_phone' => str_replace('+91','',$data['mother_phone']),
                            'guardian_phone'=> str_replace('+91','',$data['guardian_phone']),
                        ]);

                        $lastId = Student::lockForUpdate()->max('id') ?? 0;
                        $admissionNo = 'ADM-' . date('Y') . '-' . str_pad($lastId + 1, 5, '0', STR_PAD_LEFT);

                        $student = Student::create([
                            'name'         => $data['name'],
                            'admission_no' => $admissionNo,
                            'dob'          => date('Y-m-d', strtotime($data['dob'])),
                            'gender'       => strtoupper(trim($data['gender'])),
                            'category'     => $data['category'],
                            'address'      => $data['address'],
                            'parent_id'    => $parent->id,
                            'status'       => 1,
                        ]);

                        $roll = StudentClassDetail::where([
                            'class_id' => $classId,
                            'section_id' => $sectionId,
                            'financial_year_id' => $financialYearId,
                            'status' => 1,
                        ])->max('roll_no');

                        StudentClassDetail::create([
                            'student_id'        => $student->id,
                            'class_id'          => $classId,
                            'section_id'        => $sectionId,
                            'roll_no'           => $roll ? $roll + 1 : 1,
                            'financial_year_id' => $financialYearId,
                            'admission_date'    => date('Y-m-d', strtotime($data['admission_date'])),
                            'start_date'        => now(),
                            'status'            => 1,
                        ]);
                    });
                } catch (Throwable $e) {
                    $this->failRow($uploadId, $rowNumber, $data, $e->getMessage());
                    $hasErrors = true;
                }

                $this->increment($uploadId);
            }
        } catch (Throwable $e) {
            //$this->failRow($uploadId, $rowNumber, $data, $e->getMessage());
            $hasErrors = true;
        }
        DB::table('student_bulk_uploads')
        ->where('id', $uploadId)
        ->update([
            'status' => $hasErrors ? 'completed_with_errors' : 'completed',
            'updated_at' => now(),
        ]);
    }

    /* =====================================================
     | HELPERS
     ===================================================== */

    private function increment(int $uploadId): void
    {
        DB::table('student_bulk_uploads')
            ->where('id', $uploadId)
            ->increment('processed_rows');
    }

    private function failRow(int $uploadId, int $row, array $data, string $reason): void
    {
        DB::table('student_bulk_failed')->insert([
            'upload_id' => $uploadId,
            'row_number'=> $row,
            'data'      => json_encode($data),
            'reason'    => substr($reason, 0, 1000),
            'created_at'=> now(),
        ]);
    }

    private function isBlankRow(array $row): bool
    {
        foreach ($row as $value) {
            if (trim((string) $value) !== '') return false;
        }
        return true;
    }

    private function mapRow(array $row): array
    {
        $mapped = [];
        foreach ($this->headerMap as $index => $field) {
            if ($field === 'sl_no') continue;
            $mapped[$field] = trim((string) ($row[$index] ?? null)) ?: null;
        }
        return $mapped;
    }

    private function validator(array $data)
    {
        return Validator::make(
            $data,
            [
                'admission_date' => [
                    'required',
                    'date',
                    'after_or_equal:' . Carbon::now()->subMonths(3)->format('Y-m-d'),
                    'before_or_equal:' . Carbon::today()->format('Y-m-d'),
                ],

                'dob' => [
                    'required',
                    'date',
                    'before_or_equal:' . now()->subYears(2)->format('Y-m-d'),
                ],

                'name'         => 'required|string|max:255',
                'gender'       => 'required|in:Male,Female,MALE,FEMALE',
                'category'     => 'required|in:GEN,OBC,SC,ST',
                'address'      => 'required|string|max:255',
                'father_name'  => 'required|string|max:255',
                'mother_name'  => 'required|string|max:255',

                'father_phone' => [
                    'required',
                    'regex:/^(\+91)?[6-9]\d{9}$/'
                ],
            ],
            [
                // ✅ CUSTOM MESSAGES (optional but recommended)
                'father_phone.required' => 'Mobile number is required',
                'father_phone.regex'    => 'Mobile number must be a valid Indian number',
            ],
            [
                // ✅ ATTRIBUTE NAMES (THIS IS WHAT YOU WANT)
                'father_phone' => 'Mobile No',
            ]
        );

    }


    private function excelDateToDate($value): ?string
    {
        if (!$value) return null;

        if (is_numeric($value)) {
            return ExcelDate::excelToDateTimeObject($value)->format('Y-m-d');
        }

        return date('Y-m-d', strtotime($value));
    }

    /* =====================================================
     | TEMPLATE + HISTORY
     ===================================================== */

    public function history()
    {
        $uploads = DB::table('student_bulk_uploads')
            ->orderByDesc('id')
            ->get();

        return view('students.bulk-history', compact('uploads'));
    }

    public function downloadExcelTemplate()
    {
        return Excel::download(
            new StudentTemplateExport(
                array_keys($this->bulkHeaderMap()),
                $this->studentTemplateExample(),
                $this->bulkHeaderMap()
            ),
            'student_bulk_template.xlsx'
        );
    }

    private function bulkHeaderMap(): array
    {
        return [
            'SL NO'            => 'sl_no',
            'ADMISSION DATE'   => 'admission_date',
            'STUDENT NAME'     => 'name',
            'FATHER NAME'      => 'father_name',
            'MOTHER NAME'      => 'mother_name',
            'ADDRESS'          => 'address',
            'GENDER'           => 'gender',
            'DOB'              => 'dob',
            'CATEGORY'         => 'category',
            'MOBILE NO'        => 'father_phone',
        ];
    }

    private function studentTemplateExample(): array
    {
        return [
            '1',
            '10.01.2026',
            'Test Student',
            'Test Father',
            'Test Mother',
            'Habra',
            'Male',
            '08.09.1990',
            'GEN',
            '8796541230',
        ];
    }
}
